<?php
// antibot.php
class AntiBot {
    private $config;
    private $decision = array(
        'score' => 0,
        'reasons' => array(),
        'action' => 'allow'
    );

    public function __construct($config = array()) {
        $this->config = array_merge($this->defaultConfig(), $config);
    }

    public function enforce() {
        $this->runChecks();
        $this->decideAction();
        $this->logDecision();
        return $this->decision;
    }

    private function defaultConfig() {
        return array(
            'ua_required' => true,
            'ua_keywords_block' => array('bot','crawler','spider','sentinel','curl','python','wget','headlesschrome','phantomjs'),
            'ua_known_browsers' => array('Mozilla/5.0','Chrome/','Safari/','Firefox/','Edg/','OPR/'),
            'ip_blacklist' => array(),
            'cidr_blacklist' => array(),
            'org_ban_keywords' => array('amazon','aws','google','azure','digitalocean','ovh','hetzner','linode','vultr','contabo'),
            'isp_allow_keywords' => array('mtn','glo','airtel','9mobile'),
            'score_block' => 70,
            'score_challenge' => 40,
            'weight_ua_missing' => 30,
            'weight_ua_keyword' => 25,
            'weight_ua_invalid' => 15,
            'weight_ip_blacklist' => 100,
            'weight_cidr_blacklist' => 80,
            'weight_org_ban' => 40,
            'weight_non_residential_hint' => 25,
            'log_enabled' => true,
            'log_path' => dirname(__FILE__) . '/antibot.log',
        );
    }

    private function runChecks() {
        $ua = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
        $ip = $this->clientIP();

        if ($this->config['ua_required'] && trim($ua) === '') {
            $this->add('weight_ua_missing', 'UA missing');
        }

        $uaLower = strtolower($ua);
        foreach ($this->config['ua_keywords_block'] as $kw) {
            if (strpos($uaLower, $kw) !== false) {
                $this->add('weight_ua_keyword', "UA keyword: $kw");
            }
        }

        $looksBrowser = false;
        foreach ($this->config['ua_known_browsers'] as $sig) {
            if (strpos($ua, $sig) !== false) { $looksBrowser = true; break; }
        }
        if (!$looksBrowser && $ua !== '') {
            $this->add('weight_ua_invalid', 'UA not resembling common browsers');
        }

        if (in_array($ip, $this->config['ip_blacklist'])) {
            $this->add('weight_ip_blacklist', "IP blacklisted: $ip");
        }
        foreach ($this->config['cidr_blacklist'] as $cidr) {
            if ($this->cidrMatch($ip, $cidr)) {
                $this->add('weight_cidr_blacklist', "CIDR blacklisted: $cidr");
            }
        }

        $rdns = @gethostbyaddr($ip);
        $rdnsLower = strtolower($rdns ? $rdns : '');
        foreach ($this->config['org_ban_keywords'] as $kw) {
            if ($kw !== '' && strpos($rdnsLower, $kw) !== false) {
                $this->add('weight_org_ban', "Org hint: $kw via rDNS: $rdns");
            }
        }
        foreach ($this->config['isp_allow_keywords'] as $kw) {
            if ($kw !== '' && strpos($rdnsLower, $kw) !== false) {
                $this->decision['score'] = max(0, $this->decision['score'] - 10);
                $this->decision['reasons'][] = "ISP allow: $kw";
            }
        }
        foreach (array('vps','server','dedicated','cloud','static') as $kw) {
            if ($rdnsLower !== '' && strpos($rdnsLower, $kw) !== false) {
                $this->add('weight_non_residential_hint', "Non-residential hint: $kw via rDNS: $rdns");
            }
        }
    }

    private function decideAction() {
        $score = $this->decision['score'];
        if ($score >= $this->config['score_block']) {
            $this->decision['action'] = 'block';
        } elseif ($score >= $this->config['score_challenge']) {
            $this->decision['action'] = 'challenge';
        }
    }

    private function add($weightKey, $reason) {
        $weight = isset($this->config[$weightKey]) ? $this->config[$weightKey] : 0;
        $this->decision['score'] += $weight;
        $this->decision['reasons'][] = $reason;
    }

    private function clientIP() {
        $ip = isset($_SERVER['HTTP_X_FORWARDED_FOR']) ? $_SERVER['HTTP_X_FORWARDED_FOR'] : (isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '0.0.0.0');
        if (strpos($ip, ',') !== false) { $ip = trim(explode(',', $ip)[0]); }
        return $ip;
    }

    private function cidrMatch($ip, $cidr) {
        if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) return false;
        $parts = explode('/', $cidr);
        if (count($parts) != 2) return false;
        $subnet = $parts[0];
        $mask = intval($parts[1]);
        $ipLong = ip2long($ip);
        $subnetLong = ip2long($subnet);
        $maskLong = -1 << (32 - $mask);
        return ($ipLong & $maskLong) === ($subnetLong & $maskLong);
    }

    private function logDecision() {
        if (!$this->config['log_enabled']) return;
        $line = sprintf("[%s] ip=%s ua=\"%s\" score=%d action=%s reasons=%s\n",
            date('c'),
            $this->clientIP(),
            isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '',
            $this->decision['score'],
            $this->decision['action'],
            implode('; ', $this->decision['reasons'])
        );
        @file_put_contents($this->config['log_path'], $line, FILE_APPEND);
    }
}
